#include <ctype.h>
#include <string.h>
#include <stdlib.h>
#include "string_utils.h"
#include <stdio.h>
//#include "safestring.h"

void LocalConvertToUpper(char *str)
{
    char *TmpPtr;

    for (TmpPtr = str; *TmpPtr != 0; TmpPtr++)
    {
        *TmpPtr = toupper(*TmpPtr);
        if ( *TmpPtr == ' ')
            break;
    }
}

void LocalConvertEntireStringToUpper(char *str)
{
    char *TmpPtr;

    for (TmpPtr = str; *TmpPtr != '\0'; TmpPtr++)
        *TmpPtr = toupper(*TmpPtr);
}

/*-----------------10/25/01 12:15PM-----------------
 * Function:        ConvertIPAddressToString
 *
 * Return Type:     char *
 *
 * Return Values:   number of data elements in the pipe
 *
 * Purpose:         This routine will convert a IP address into dot.decimal notation.
 *
 * Parameters:      unsigned int IPAddress  - address to convert.
 *                  char *AddressString     - string to contain converted address
 * --------------------------------------------------*/
char *ConvertIPAddressToString(unsigned int IPAddress, char *AddressString)
{
    unsigned char *Ptr = (unsigned char *)&IPAddress;

    sprintf(AddressString, "%d.%d.%d.%d", Ptr[0], Ptr[1], Ptr[2], Ptr[3]);

    return AddressString;
}

// for places where string output needs to be parsed
char *ConvertIPAddressToStringZeropad(unsigned int IPAddress, char *AddressString, UINT32 addrStrLen)
{
    unsigned char *Ptr = (unsigned char *)&IPAddress;

    snprintf(AddressString, addrStrLen, "%d.%d.%d.%d", Ptr[0], Ptr[1], Ptr[2], Ptr[3]);

    return AddressString;
}

/*-----------------8/3/01 5:05PM--------------------
 * Function:        ConvertStringToIPAddress
 *
 * Return Type:     unsigned long
 *
 * Return Values:   the address represented by the string in dot decimal format.
 *
 * Description:     This will convert the string in dot decimal format to an unsigned long
 *                  address.
 *
 * Parameters:      char* AddressString - pointer to string containing address.
 * --------------------------------------------------*/
unsigned long ConvertStringToIPAddress(char* AddressString)
{
    unsigned char NewAddress[4];
    unsigned int  LoopCnt = 0;
    char *EndPtr,*NextGroup;
    unsigned long tmp;

    NextGroup = strtok(AddressString," -.,;:");
    for ( LoopCnt = 0; LoopCnt < 4; LoopCnt++)
    {
        if ( NextGroup == NULL )       // check if there is enough input data
            return 0;

        tmp = strtoul(NextGroup, &EndPtr, 10);
        if ((*EndPtr != '\0') || (tmp > 255))
            return 0;

        NewAddress[LoopCnt] = (unsigned char)tmp;

        NextGroup = strtok(NULL," -.,;:");
    }

    return *((unsigned long *)NewAddress);
}

char *RemoveWhiteSpace(char *StringPtr)
{
    while ( (*StringPtr != '\0') && isspace(*StringPtr) )
    {
        StringPtr++;
    }

    return StringPtr;
}

// replace "" in a string with "
// does it in-place
void StringUnescapeQuotes(char * p)
{
    int len;
    //int i;

    if(!p)
    {
        return;
    }

    while(*p)
    {
        if(*p == '"')
        {
            if(*(p+1) == '"')
            {
                len = strlen(p);
                memmove(p, p+1, len);
            }
        }
        p++;
    }
}

/**
 * \author      Pete McCormick
 *
 * \date        05/10/2005
 *
 * \return      UINT32
 *
 * \retval      the number of array elements created
 *
 * \brief       Takes a line of a comma-delimited strings
 *              and creates an an array of pointers to them.
 *
 * \param       fieldNames
 * \param       fieldNameCnt
 * \param       linebuf
 *
 * \note        quotes within field name get doubled
 * \note        empty fields (without quotes) are allowed
 * \note        commas, CR/LF within quoted field are allowed
 * \note        either carriage-return or linefeed can indicate end of line
 *
 * \warning     fields must be quoted or empty - otherwise can't tell if a
 *              comma belongs in the field or is a delimiter
 */
UINT32 StringCommaDelimitedToArray(char * fieldNames[], UINT32 fieldNameCnt, char * linebuf)
{
    UINT32 i;
    UINT32 step = 0;
    UINT32 fields = 0;
    //int len;
    char *p;
    char * pField = linebuf;

    for(i=0; i<fieldNameCnt; i++)
    {
        fieldNames[i] = 0;
    }

    p = linebuf;
    while(*p)
    {
        switch(step)
        {
            case 0: // outside a field
                if(*p == ',')
                {
                    *p = 0;
                    // empty field
                    if(fields >= fieldNameCnt)
                    {
                        return fields;
                    }
                    fieldNames[fields++] = p;
                    p++;
                }
                else if(*p == '"')
                {
                    p++;    // skip the field-opening quote
                    pField = p;
                    step = 1;
                }
                else
                {
                    p++;
                }
                break;

            case 1: // inside a field
                if(*p == '"')
                {
                    // field-ending quote or double quote?
                    if(*(p+1) == '"')
                    {
                        *p = ' ';   // replace escape quote with a space
                        p++;    // skip space
                        p++;    // skip quote
                    }
                    else if((*(p+1) == 0) || (*(p+1) == '\r') || (*(p+1) == '\n'))
                    {
                        // found an end-field quote, no more fields
                        if(fields >= fieldNameCnt)
                        {
                            return fields;
                        }
                        fieldNames[fields++] = pField;
                        *p++ = 0;    // replace and skip quote
                        *p = 0;      // replace \r or \n with null
                        return fields;
                    }
                    else if(*(p+1) == ',')
                    {
                        // found an end-field quote, possibly more fields
                        if(fields >= fieldNameCnt)
                        {
                            return fields;
                        }
                        fieldNames[fields++] = pField;
                        *p++ = 0;   // replace and skip quote
                        *p++ = 0;   // replace comma with null
                        step = 0;
                    }
                }
                else
                {
                    p++;
                }
                break;

//            default:
//                p++;
//                break;
        }
    }

// strtok doesn't work because it can't handle empty fields
// AND embedded commas are allowed!

//     p = strtok(linebuf, ",");
//     while(p)
//     {
//         // skip leading quote, if any
//         if(*p == '"')
//         {
//             p++;
//         }
//
//         if(*p)
//         {
//             // remove trailing quote, if any
//             len = strlen(p);
//             if(p[len-1] == '"')
//             {
//                 p[len-1] = 0;
//             }
//         }
//         StringUnescapeQuotes(p);
//         if(fields >= fieldNameCnt)
//         {
//             break;
//         }
//         fieldNames[fields++] = p;
//         p = strtok(NULL, ",");
//     }

    return fields;
}

/**
 * \author      Pete McCormick
 *
 * \date        05/10/2005
 *
 * \return      char *
 *
 * \retval
 *
 * \brief       Allows caller to find a string value for the matching key value.
 *
 * \param       key - the key of the desired value
 * \param       keys - array of keys
 * \param       values - array of values corresponding to keys
 * \param       valuesCnt - number of key,value pairs in the array
 *
 * \warning     Not efficient at all, but acceptable if valuesCnt is small.
 *
 */

char * StringLookupByString(char * key, char * keys[], char * values[], UINT32 valuesCnt)
{
    UINT32 i;

    if(!key)
    {
        return NULL;
    }

    for(i=0; i<valuesCnt; i++)
    {
        if(!keys[i])
        {
            continue;
        }
        if(strcmp(key, keys[i]) == 0)
        {
            return(values[i]);
        }
    }

    return NULL;
}

/**
 * \author      S.Novick
 * \date        04/01/2013
 * \brief       Takes a MAC address string of the form xx.xx.xx.xx.xx.xx
 *              and converts it a to 6-byte array. Other reasonable delimiters allowed.
 * \param       p - pointer to the input MAC address string
 * \param       NewAddress - pointer to the output MAC address byte array
 * \return      INT32
 * \retval      0 - success, -1 = failure
 * \warning     Calling function must allocate space for the MAC address byte
 */
INT32 ConvertStringToMacAddress(UINT8 * NewAddress, char * p)
{
    unsigned char mac[6];
    int i;
    char *EndPtr, *NextGroup;
    unsigned long tmp;

    NextGroup = strtok(p," -.,;:");

    for (i = 0; i < 6; i++)
    {
        if (NextGroup == NULL)       // check to see if there is enough input data
            return -1;

        tmp = strtoul(NextGroup, &EndPtr, 16);
        if ((*EndPtr != '\0') || (tmp > 255))
            return -1;

        mac[i] = (unsigned char)tmp;

        NextGroup = strtok(NULL," -.,;:");
    }

    for (i = 0; i < 6; i++)
    {
        NewAddress[i] = mac[i];
    }

    return 0;
}

/**
 * \author      Pete McCormick
 *
 * \date        07/24/04
 *
 * \return      char *
 *
 * \retval      pointer to MAC address string
 *
 * \brief       Takes a MAC address 6-byte array and converts it a
 *              string of the form xx.xx.xx.xx.xx.xx.
 *
 * \param       p - pointer to the MAC address string
 * \param       NewAddress - pointer to the MAC address byte array
 *
 * \warning     Calling function must allocate space for the MAC address string.
 *
 */
char * ConvertMacAddressToString(char * p, UINT8 * NewAddress)
{
    unsigned int temp[6];
    int i;

    for(i=0; i<6; i++)
    {
        temp[i] = (unsigned int)NewAddress[i];
    }

    sprintf(p, "%02x.%02x.%02x.%02x.%02x.%02x",
            temp[0], temp[1], temp[2], temp[3], temp[4], temp[5]);

    return p;
}

#define NUM_HOURS    (60*60)
#define NUM_MINS     (60)

char * Seconds2HMS(char *DestString, UINT32 len, UINT32 seconds)
{
    int Hours, Mins, Secs;

    Hours = seconds / NUM_HOURS;
    seconds = seconds % NUM_HOURS;
    Mins = seconds / NUM_MINS;
    seconds = seconds % NUM_MINS;
    Secs = seconds;

    if(Hours)
    {
        snprintf(DestString, len, "%d:%02d:%02d", Hours, Mins, Secs);
    }
    else
    {
        snprintf(DestString, len, "%d:%02d", Mins, Secs);
    }
    return(DestString);
}

// change "string" to string
void StringRemoveQuotes(char * p)
{
    int len;

    if(!p)
    {
        return;
    }

    if(*p == 0)
    {
        return;
    }

    if(*p != '"')
    {
        return;
    }
    len = strlen(p);
    if(len < 2)
    {
        return;
    }
    memmove(p, p+1, len);
    p[len-2] = 0;
}

/*-----------------12/3/01 10:04AM------------------
 * Function:        ConvertTimeToHMS
 *
 * Returns:         char *
 *
 * Return Values:   pointer to destination string
 *
 * Description:     This routine converts the time stamp to days hh:mm:ss.ss format
 *
 * Parameters:      unsigned long Timestamp - number of ticks since bootup in 1/100 s
 *                  char *DestString - pointer to destination
 * Reentrant
 * --------------------------------------------------*/
#define NUM_DAYS_TICKS     (24*60*60*100)
#define NUM_HOURS_TICKS    (60*60*100)
#define NUM_MINS_TICKS     (60*100)
#define NUM_SECS_TICKS     (100)

char *ConvertTimeToHMS(UINT32 Timestamp, char *DestString)
{
    int Days, Hours, Mins, Secs;

    Days = Timestamp / NUM_DAYS_TICKS;
    Timestamp = Timestamp % NUM_DAYS_TICKS;
    Hours = Timestamp / NUM_HOURS_TICKS;
    Timestamp = Timestamp % NUM_HOURS_TICKS;
    Mins = Timestamp / NUM_MINS_TICKS;
    Timestamp = Timestamp % NUM_MINS_TICKS;
    Secs = Timestamp / NUM_SECS_TICKS;
    Timestamp = Timestamp % NUM_SECS_TICKS;

    sprintf(DestString, "%d days %02d:%02d:%02d.%02d", (int)Days, (int)Hours, (int)Mins, (int)Secs, (int)Timestamp);
    return(DestString);
}

/**
 * \author      Pete McCormick
 *
 * \date        01/31/2006
 *
 * \return      INT32
 *
 * \retval      0: success
 * \retval      <0: error
 *
 * \brief       Looks up a string in a list and outputs the corresponding number,
 *              if any.
 *
 * \param       pPairs
 * \param       pStr
 * \param       pNum
 *
 * \note        This is very inneficient, but works well for small lists.
 *
 * \warning     ppPairs must have a final element with a NULL string pointer,
 *              in order to terminate the list.
 */
INT32 StringGetNumber(STRNUMPAIR * pPairs, char * pStr, UINT32 * pNum,
                      BOOL caseSensitive)
{
    UINT32 i;
    INT32 result;

    for(i=0; pPairs->pStr; i++, pPairs++)
    {
        if(caseSensitive)
        {
            result = strcmp(pStr, pPairs->pStr);
        }
        else
        {
            result = stricmp(pStr, pPairs->pStr);
        }
        if(result == 0)
        {
            *pNum = pPairs->num;
            return 0;
        }
    }
    return -1;
}



/**
 * \author      Pete McCormick
 *
 * \date        01/31/2006
 *
 * \return      INT32
 *
 * \retval      0: success
 * \retval      <0: error
 *
 * \brief       Looks up a number in a list and outputs the corresponding string,
 *              if any.
 *
 * \param       pPairs
 * \param       num
 * \param
 *
 * \note        This is very inefficient, but works well for small lists.
 *
 * \warning     ppPairs must have a final element with a NULL string pointer,
 *              in order to terminate the list.
 */

INT32 NumberGetString(STRNUMPAIR * pPairs, UINT32 num, char * pDest, UINT32 destLen)
{
    UINT32 i;

    for(i=0; pPairs->pStr; i++, pPairs++)
    {
        if(pPairs->num == num)
        {
            strncpy(pDest, pPairs->pStr, destLen);
            return 0;
        }
    }
    *pDest = 0;
    return -1;
}

/**
 * \author      Pete McCormick
 *
 * \date        01/31/2006
 *
 * \return      INT32
 *
 * \retval      0: success
 * \retval      <0: error
 *
 * \brief       Parses a command line with multiple arguments and populates
 *              an array of pointers to arguments.
 *
 * \param       cmd
 * \param       argv
 * \param       maxArgs
 * \param       pArgc
 *
 * \note        Arguments are delimited by spaces.
 *              Quoted strings are allowed in order to have spaces within
 *              arguments.
 *
 * \warning
 *
 */
INT32 StringParseArgs(char * cmd, char * argv[], UINT32 maxArgs, UINT32 * pArgc)
{
    char * p;
    UINT32 arg;

    for(arg = 0; arg<maxArgs; arg++)
    {
        argv[arg] = 0;
    }

    arg = 0;
    p = cmd;
    while(arg < maxArgs)
    {
        if(*p == 0)
        {
            break;
        }
        // seek 1st non-space
        while(*p && (*p == ' '))
        {
            p++;
        }
        if(*p == 0)
        {
            break;
        }
        // not at a space
        if(*p == '"')
        {
            // skip quote
            p++;
            if(*p == 0)
            {
                break;
            }
            if(arg >= maxArgs)
            {
                break;
            }
            argv[arg++] = p;
            // look for next quote
            while(*p && (*p != '"'))
            {
                p++;
            }
            if(*p == 0)
            {
                break;
            }
            // assume at quote
            *p = 0;
            // skip past quote
            p++;
            if(*p == 0)
            {
                break;
            }
            // skip spaces, if any
            while(*p && (*p == ' '))
            {
                p++;
            }
            if(*p == 0)
            {
                break;
            }
            // assume not at space
        }
        else
        {
            // not a quote
            if(arg >= maxArgs)
            {
                break;
            }
            argv[arg++] = p;
            // seek next space
            while(*p && (*p != ' '))
            {
                p++;
            }
            if(*p == 0)
            {
                break;
            }
            // must be at a space - whack it and move on!
            *p = 0;
            p++;
        }
    }

    *pArgc = arg;

    return 0;
}

int stricmp(char const *s1, char const *s2)
{
  char f, l;

  do
  {
    f = ((*s1 <= 'Z') && (*s1 >= 'A')) ? *s1 + 'a' - 'A' : *s1;
    l = ((*s2 <= 'Z') && (*s2 >= 'A')) ? *s2 + 'a' - 'A' : *s2;
    s1++;
    s2++;
  } while ((f) && (f == l));

  return (int) (f - l);
}

UINT8 AsciiHexToNumber(char * p)
{
    UINT8 data = 0;
    //UINT32 i;

    LocalConvertEntireStringToUpper(p);

    if((*p >= 'A') && (*p <= 'F'))
    {
        data += (*p - 'A' + 10);
    }
    else if((*p >= '0') && (*p <= '9'))
    {
        data += (*p - '0');
    }
    p++;
    if(*p == 0)
    {
        return data;
    }
    data <<= 4;
    if((*p >= 'A') && (*p <= 'F'))
    {
        data += (*p - 'A' + 10);
    }
    else if((*p >= '0') && (*p <= '9'))
    {
        data += (*p - '0');
    }

    return data;
}


/**
 * \author      Pete McCormick (copied from Diab)
 *
 * \date        8/18/2006
 *
 * \return      char *
 *
 * \retval      pointer to dest string
 *
 * \brief       copies one string to another.
 *
 * \param       s1 : dest string
 * \param       s2 : src string
 * \param       n : max byte count
 *
 * \note        Made this version of strncpy so that it does not write trailing
 *              nulls, to avoid bugs where a string buffer gets overrun.
 *
 */
//char *strncpy(char *s1, const char *s2, size_t n)
//{
//	char *ret = s1;
//    int x = n;
//    //int i;
//
//    if (!s1 || !n)
//    {
//        return ret;
//    }
//
//    while(x--)
//    {
//        if ((*s1++ = *s2++) == '\0')
//        {
//			break;
//		}
//	}
//    // always NULL-terminate dest
//    if(x < 0)
//    {
//        ret[n-1] = 0;
//    }
//
//	return ret;
//}

/**
 * \author      S.Novick
 * \date        04/02/2013
 * \brief       copies one string to another.
 * \param       dest string
 * \param       src string
 * \param       n : max bytes to copy
 * \return      char *
 * \retval      pointer to dest string
 * \note        Safer version of strncpy, always null terminates the string.
 */
char* _strncpy(char *dest, const char *src, int n)
{
    if (dest && (n > 0))
    {
        strncpy(dest, src, n-1);
        dest[n-1] = '0';
    }
    return dest;
}

/**
 * \author      S.Novick
 * \date        04/03/2013
 * \brief       appends one string to another.
 * \param       dest string
 * \param       src string
 * \param       n : max bytes destination can hold
 * \return      char *
 * \retval      pointer to dest string
 * \note        Safer version of strncat, always appends, truncates if necessary, null-terminates.
 */
char* _strncat(char *dest, const char *src, int n)
{
    if (dest && (n > 0))
    {
        int len = strlen(dest);
        _strncpy(dest + len, src, n - len);
    }
    return dest;
}

/**
 * \author      Pete McCormick (copied from Diab)
 *
 * \date        8/22/2006
 *
 * \return      char *
 *
 * \retval      pointer to dest string
 *
 * \brief       Trims any number of the specified character from the end of the
 *              string.
 *
 * \param       pDest
 * \param       trimChar
 *
 * \warning     Modifies the string passed in.
 *
 */
char * StringTrim(char * pDest, char trimChar)
{
    if (pDest)
    {
        // point to last char in string
        char* p = pDest + strlen(pDest) - 1;

        while(p >= pDest)
        {
            if(*p != trimChar)
            {
                *(p+1) = 0;
                break;
            }
            p--;
        }
    }

    return pDest;
}

char * StringReplace(char * pDest, char oldChar, char newChar)
{
    if (pDest)
    {
        char* p = pDest;

        while(*p)
        {
            if(*p == oldChar)
            {
                *p = newChar;

            }
            p++;
        }
    }

    return pDest;
}

/**
 * \author      Pete McCormick (copied from Diab)
 *
 * \date        8/22/2006
 *
 * \return      char *
 *
 * \retval      pointer to dest string
 *
 * \brief       Returns a substring of pSrc string that is len chars long.
 *
 * \param       pDest
 * \param       pSrc
 * \param       startIdx
 * \param       len
 *
 * \note        If pSrc is an empty string,
 *              pDest will become an empty string also.
 *
 */
char * StringSub(char * pDest, char * pSrc, UINT32 startIdx, UINT32 len)
{
	char *p;
	char *q = pDest;
	if(startIdx >= strlen(pSrc))
	{
        *q = 0;
        return q;
	}
	for(p=&pSrc[startIdx]; *p && len--; p++)
	{
		*pDest++ = *p;
	}
	*pDest = 0;

	return q;
}

int strpos (char * s, char c)
{
    char *reply;

    if ((reply = strchr (s, c))==NULL) {
        return -1;
    }
    return (reply - s);
}

char * StringFixedPointFormat(char * pDest, UINT32 destChars, INT32 val, INT32 div)
{
    if(val == 0)
    {
        snprintf(pDest, destChars, "%ld.%ld", val/div, val%div);
    }
    else if(val > 0)
    {
        snprintf(pDest, destChars, "+%ld.%ld", val/div, val%div);
    }
    else                                // <0
    {
        val *= -1;
        snprintf(pDest, destChars, "-%ld.%ld", val/div, val%div);
    }
    return pDest;
}

void StringPad(char * pSrc, char c, UINT32 len)
{
    UINT32 charCnt = strlen(pSrc);
    char * p;

    for(p=&pSrc[charCnt]; p<pSrc+len; p++)
    {
        *p = c;
    }
    *p = 0;
}

/**
 * \author      Pete McCormick (copied from Diab)
 *
 * \date        5/24/2007
 *
 * \return      char *
 *
 * \retval      pointer to dest string
 *
 * \brief       Outputs a block of binary data as ascii hex.
 *
 * \param       pDest
 * \param       pSrc
 * \param       byteCnt
 *
 * \warning     Caller must allocate at least 2*byteCnt + 1 bytes for pDest
 */
char * StringBin2Hex(char * pDest, UINT8 * pSrc, UINT32 byteCnt)
{
    char * p = pDest;

    while(byteCnt--)
    {
        p += snprintf(p, 4, "%02X", *pSrc++);
    }
    *p = 0;

    return pDest;
}

BOOL StringCharIsAlphanumeric(char c)
{
    if((c >= 'a') && (c <= 'z'))
    {
        return 1;
    }
    if((c >= 'A') && (c <= 'Z'))
    {
        return 1;
    }
    if((c >= '0') && (c <= '9'))
    {
        return 1;
    }
    return 0;
}

UINT16 Swap16(UINT16 sval)
{
#if 0 //S.N 08/23/2013
    UINT8 TempVal;
    UINT16 RetVal;

    TempVal = (sval & 0xFF00) >> 8;
    RetVal = ((sval & 0x00FF) << 8) | TempVal;

    return RetVal;
#endif
    return ((sval << 8) | (sval >> 8 )); 
}

UINT32 Swap32(UINT32 sval)
{
    UINT32 TempVal;
    UINT32 RetVal=0;

    TempVal = (sval & 0xFF000000) >> 24;
    RetVal |= TempVal;
    TempVal = (sval & 0x00FF0000) >> 8;
    RetVal |= TempVal;
    TempVal = (sval & 0x0000FF00) << 8;
    RetVal |= TempVal;
    TempVal = (sval & 0x000000FF) << 24;
    RetVal |= TempVal;

    return RetVal;
}

/**
 *
 * \author      Hazrat Shah
 *
 * \brief		This function converts ASCII char to HEX
 *
 * \detail
 *
 * \date        05/10/2007
 *
 * \param	    INT8 ch, UINT8 *nib
 *
 * \return		BOOL
 *
 * \retval		Returns TRUE if successful
 */
BOOL ConvertASCIItoHexnib (INT8 ch, UINT8 *nib) {
	BOOL res = FALSE;

	if 			   ((ch >= 0x30) && (ch <= 0x39)) 	{
		*nib = ch-0x30;		res = TRUE;
	}	else	if ((ch >= 0x41) && (ch <= 0x46))	{
		*nib = ch-0x41+0xA;	res = TRUE;
	}	else	if ((ch >= 0x61) && (ch <= 0x66))	{
		*nib = ch-0x61+0xA;	res = TRUE;
	}

	return (res);
}

/**
 *
 * \author      Hazrat Shah
 *
 * \brief		This function convets ASCII HEX to HEX
 *
 * \detail
 *
 * \date        05/10/2007
 *
 * \param	    INT8 *psrc, UINT8 *pdata, UINT8 pdatalen
 *
 * \return		BOOL
 *
 * \retval		Returns TRUE if successful
 */
BOOL	ConvertASCIIStringtoHex (INT8 *psrc, UINT8 *pdata, UINT8 pdatalen)	{
	BOOL	res = TRUE;
	UINT8	i, nib;

	//	Make sure the string has numerical values, the string should always contains HEX values
	for (i=0; i<(pdatalen*2) && (*psrc) && (*psrc != ' ') && (*psrc != '\r') && (*psrc != '\n'); i++, *psrc++)	{
		if (ConvertASCIItoHexnib (*psrc, &nib)) {
			*(pdata+(i/2)) <<= 4; *(pdata+(i/2)) |= nib;
		}	else	{
			//	Unsuccessful conversion, fail
			res = FALSE;
			break;
		}
	}

	return (res);
}

/**
 *
 * \author      Hazrat Shah
 *
 * \brief		This function breaks the passed on strings into smaller pieces using the delimiters " .,;:"
 *				ASCII hex are converted to HEX and DECIMAL values
 *
 * \detail
 *
 * \date        05/10/2007
 *
 * \param	    INT8 *psrc, TYPE_ARG_STRUCT *pargv, UINT8 *argc, UINT8 maxarg
 *
 * \return		None
 *
 * \retval		None
 */
void	GetCmdArguments (INT8 *psrc, TYPE_ARG_STRUCT *pargv, UINT8 *argc, UINT8 maxarg)	{
	INT8  *endptr;
    UINT8 arg = 0;
	*argc = 0;

	//	clear memory
	memset ((UINT8*)pargv, 0, sizeof(TYPE_ARG_STRUCT)*maxarg);

	//	separate arguments
	pargv[arg].argv = (INT8*) strtok((char*)psrc, " -.,;:");
	if (pargv[arg].argv)	{	LocalConvertToUpper((char*)pargv[arg].argv);		arg +=1;	}
	while (arg < maxarg)	{
		pargv[arg].argv = (INT8*) strtok(NULL, " -.,;:");
		if (pargv[arg].argv){	LocalConvertToUpper((char*)pargv[arg].argv);		arg +=1;	}	else	{	break;	}
	}
	*argc = arg;

	//	convert to HEX
	arg = 0;
	while (arg < *argc) {
		if ((pargv[arg].len  = strlen((char const*)pargv[arg].argv))) {
			if (ConvertASCIIStringtoHex (pargv[arg].argv, pargv[arg].parlng, (pargv[arg].len>MAX_CONSOLE_PAR_LNG_LEN)?MAX_CONSOLE_PAR_LNG_LEN:pargv[arg].len)) {
				pargv[arg].type = CONSOLE_PAR_TYPE_NUM;
				//memcpy ((UINT8*)&pargv[arg].parh, (UINT8*)&pargv[arg].parlng[0], sizeof(pargv[arg].parh));
				//memcpy ((UINT8*)&pargv[arg].parl, (UINT8*)&pargv[arg].parlng[4], sizeof(pargv[arg].parl));
				pargv[arg].parh   = strtoul ((char const*)pargv[arg].argv, (char**)&endptr, 16);
				pargv[arg].pardec = strtoul ((char const*)pargv[arg].argv, (char**)&endptr, 10);
			}	else	{
				pargv[arg].type = CONSOLE_PAR_TYPE_STR;
			}
		}
		arg++;
	}
}

/**
 *
 * \author      Hazrat Shah
 *
 * \brief		This function adjust the string width to given size, for shorter
 *				strings space chars are added, longer strings are trimmed to the
 *				passed on string.
 *
 *
 * \detail
 *
 * \date        05/21/2007
 *
 * \param	    INT8 *psrc, UINT8 *pdst, UINT8 pdatalen
 *
 * \return		None
 *
 * \retval		None
 */
void	StringAdjustToWidth (INT8 *psrc, INT8 *pdst, UINT8 StrWidth)	{
	UINT8	len = 0;

	while ((StrWidth > len) && (*(psrc+len)))	{
		*(pdst+len) = *(psrc+len);
		len++;
	}

	if (StrWidth > len)	{
		//	fill left over space with space char
		memset (pdst+len, ' ', StrWidth-len);
		len = StrWidth;
	}

	//	terminate string
	*(pdst+len-1) = 0;
}

/**
* \author   Hazrat Shah
* \brief    Check for help string
*
* \date     07/07/2010
* \param    *cmd- pointer to command string
*
* \return   BOOL
* \retval   Return true if successful
*/
BOOL IsCmdHelp (INT8 *pstr)	{
	BOOL res = false;

	if (pstr && ((strcmp ((char*)pstr, "?") == 0) || (strcmp ((char*)pstr, "HELP") == 0)))	{
		res = true;
	}

	return (res);
}